/* SCLoRaSend_and_Receive.ino
 *  A sketch to set up the SX-1278 chip in an RA-02 LoRa transceiver
 *  and then allow LoRa Duplex communication. We set up the RA-02 to
 *  operate on 434MHz, with a spreading factor of 8 and a transmit
 *  power of +14dBm (25mW), for legal use in Australia.
 *  Adapted from sketches by Tom Igoe by Jim Rowe, for
 *  Silicon Chip magazine. Last revised 31/01/2019 at 7:05 am
 *  
 *   Note that the sketch assumes that the RA-02 module's pins
 *   are connected to the following pins of the Arduino:
 *   RA-02 Pin(s)      Arduino Pin
 *   1,2,9,16         GND
 *   3 (3.3V)         +3.3V
 *   4 (RESET)        IO9
 *   5 (DIO0)         IO2
 *   12 (SCK)         IO13 (or pin 3 of ICSP header)
 *   13 (MISO)        IO12 (or pin 1 of ICSP header)
 *   14 (MOSI)        IO11 (or pin 4 of ISCP header)
 *   15 (NSS)         IO10
*/
#include <SPI.h>          // include SPI & LoRa libraries
#include <LoRa.h>

const int csPin = 10;     // LoRa radio chip select (SS-bar of RA-02)
const int resetPin = 9;   // LoRa radio reset (pin 4 of RA-02)
const int irqPin = 2;     // DIO0 pin of RA-02 (must be a hardware interrupt pin)

void setup()
{
  Serial.begin(9600);     // initialize serial
  while (!Serial);        // wait until serial port is awake
  Serial.println("RA-02 LoRa Send and Receive");

  // override the default CS, reset, and IRQ pins
  LoRa.setPins(csPin, resetPin, irqPin); // set CS, reset, IRQ pin

  if (!LoRa.begin(434E6))
  {             // initialize radio at 434 MHz
    Serial.println("LoRa init failed. Check your connections.");
    while (true);         // if failed, do nothing
  }
  Serial.println("Tx & Rx frequency set to 434MHz");
  
  LoRa.setSpreadingFactor(8);  // (can range from 6-12)
  Serial.println("Spreading Factor set to 8");
  LoRa.setTxPower(14, 15);   // set Tx output power to +14dBm
  Serial.println("Transmit power set to +14dBm (25mW)"); 
  Serial.println(" ");
  // next two lines are purely for testing
  // Serial.println("LoRa Register Dump:");
  // LoRa.dumpRegisters(Serial);   // fetch the registers & display
}
void loop()
{
  if (Serial.available() > 0)
  {
    String message = Serial.readString();
    sendMessage(message);
    Serial.println("Tx message sent: " + message);
  }
  // parse for a packet, and call onReceive with the result:
  onReceive(LoRa.parsePacket());
}
// ----------------------------------------------------------------------
void sendMessage(String outgoing)
{
  LoRa.beginPacket();          // start packet
  LoRa.print(outgoing);        // add payload
  LoRa.endPacket();            // finish packet and send it
}
// ----------------------------------------------------------------------
void onReceive(int packetSize)
{
  if (packetSize == 0) return; // if no packet, just return

  // read packet header bytes:
  String incoming = "";

  while (LoRa.available())
  {
    incoming += (char)LoRa.read();
  }

  Serial.println("Rx Message: " + incoming);
  Serial.println("RSSI: " + String(LoRa.packetRssi()));
  Serial.println("SNR: " + String(LoRa.packetSnr()));
  Serial.println();
}
// end of sketch -------------------------------------------------------
